/* --------------------------------------------------------------
  StyleApi.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

import Ajax from '../libs/Ajax';
import JsonParser from '../libs/JsonParser';

export default {
	/**
	 * Get style data by name.
	 *
	 * @param {String} styleName The name of the style to be retrieved.
	 *
	 * @return {jQuery.jqXHR}
	 */
	get(styleName) {
		StyleEdit.Validator.isString(styleName);
		
		const ajaxUrl = StyleEdit.Config.get('baseUrl') + '/api.php/templates/' +
			encodeURIComponent(StyleEdit.Config.get('template')) + '/styles/' +
			encodeURIComponent(styleName) + '?lang=' +
			encodeURIComponent(StyleEdit.Config.get('languageCode'));
		
		return Ajax.get(ajaxUrl, {theme: StyleEdit.Config.get('theme') || undefined}, {dataType: 'json'});
	},
	
	/**
	 * Get all style configurations.
	 *
	 * @return {jQuery.jqXHR}
	 */
	list() {
		const ajaxUrl = StyleEdit.Config.get('baseUrl') + '/api.php/templates/'
			+ encodeURIComponent(StyleEdit.Config.get('template')) + '/styles'
			+ '?lang=' + encodeURIComponent(StyleEdit.Config.get('languageCode'));
		return Ajax.get(ajaxUrl, {theme: StyleEdit.Config.get('theme') || undefined}, {dataType: 'json'});
	},
	
	/**
	 * Activate Style
	 *
	 * @param {String} styleName The name of the style to be activated.
	 *
	 * @return {jQuery.jqXHR}
	 */
	activate(styleName) {
		StyleEdit.Validator.isString(styleName);
		
		const ajaxUrl = StyleEdit.Config.get('baseUrl') + '/api.php/templates/'
			+ encodeURIComponent(StyleEdit.Config.get('template'))
			+ '/styles/' + encodeURIComponent(styleName) + '/actions/activate';
		
		return Ajax.patch(ajaxUrl, {theme: StyleEdit.Config.get('theme') || undefined}, {dataType: 'json'});
	},
	
	/**
	 * Create New Style
	 *
	 * The success callback will contain the new style data as response.
	 *
	 * @param {String} boilerplateName
	 * @param {String} newStyleName
	 *
	 * @return {jQuery.jqXHR}
	 */
	create(boilerplateName, newStyleName) {
		StyleEdit.Validator.isString(boilerplateName);
		StyleEdit.Validator.isString(newStyleName);
		
		const ajaxUrl = StyleEdit.Config.get('baseUrl') + '/api.php/templates/'
			+ encodeURIComponent(StyleEdit.Config.get('template')) + '/styles';
		const data = {
			boilerplate_name: boilerplateName,
			new_style_name: newStyleName,
			theme: StyleEdit.Config.get('theme') || undefined
		};
		
		return Ajax.post(ajaxUrl, data, {dataType: 'json'});
	},
	
	/**
	 * Save Style
	 *
	 * @param {StyleConfiguration} styleConfiguration Contains the style configuration data.
	 * @param {String} tempStyleName Optional, if provided it will replace the original style name (used for preview).
	 *
	 * @return {jQuery.jqXHR}
	 */
	save(styleConfiguration, tempStyleName) {
		StyleEdit.Validator.isObject(styleConfiguration);
		
		const styleName = tempStyleName || styleConfiguration.getName();
		const ajaxUrl = StyleEdit.Config.get('baseUrl') + '/api.php/templates/'
			+ encodeURIComponent(StyleEdit.Config.get('template')) + '/styles/'
			+ encodeURIComponent(styleName);
		const data = JsonParser.encodeStyleConfiguration(styleConfiguration, false);
		
		if (tempStyleName) {
			data.name = tempStyleName;
		}
		
		return Ajax.put(ajaxUrl, {style_json: JSON.stringify(data), theme: StyleEdit.Config.get('theme') || undefined}, {dataType: 'json'});
	},
	
	/**
	 * Delete Existing Style
	 *
	 * @param {String} styleName
	 *
	 * @return {jQuery.jqXHR}
	 */
	delete(styleName) {
		StyleEdit.Validator.isString(styleName);
		
		const ajaxUrl = StyleEdit.Config.get('baseUrl') + '/api.php/templates/'
			+ encodeURIComponent(StyleEdit.Config.get('template')) + '/styles/' + encodeURIComponent(styleName);
		
		return Ajax.delete(ajaxUrl, {theme: StyleEdit.Config.get('theme') || undefined}, {dataType: 'json'});
	},
	
	/**
	 * Duplicate Style
	 *
	 * The success callback will contain the new style data.
	 *
	 * @param {String} oldStyleName
	 * @param {String} newStyleName
	 *
	 * @return {jQuery.jqXHR}
	 */
	duplicate(oldStyleName, newStyleName) {
		StyleEdit.Validator.isString(oldStyleName);
		StyleEdit.Validator.isString(newStyleName);
		
		const ajaxUrl = StyleEdit.Config.get('baseUrl') + '/api.php/templates/'
			+ encodeURIComponent(StyleEdit.Config.get('template'))
			+ '/styles/' + encodeURIComponent(oldStyleName);
		const data = {
			new_style_name: newStyleName,
			theme: StyleEdit.Config.get('theme') || undefined
		};
		
		return Ajax.post(ajaxUrl, data, {dataType: 'json'});
	}
}
